#! /usr/bin/env python

'''
Copyright (c) 2005, WukooPy Team (lihuimail@gmail.com)
All rights reserved.

Redistribution and use in source and binary forms, with or without modification, 
are permitted provided that the following conditions are met:

    * Redistributions of source code must retain the above copyright notice, 
      this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright notice, 
      this list of conditions and the following disclaimer in the documentation 
      and/or other materials provided with the distribution.
    * Neither the name of the WukooPy Team nor the names of its contributors 
      may be used to endorse or promote products derived from this software 
      without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE 
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE 
FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR 
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER 
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, 
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
'''

"""
WSGI wrapper for ASP.


Example Global.asa for a CherryPy app called "mcontrol":

<script language=Python runat=Server> 
def Application_OnStart():
    Application.Contents("multiprocess") = False
    Application.Contents("multithread") = True
    from mcontrol import chpy
</script>


Example handler.asp:

<%@Language=Python%>
<%
from wsgiref.asp_gateway import handler
from cherrypy.wsgiapp import wsgiApp

handler(Application, Request, Response).run(wsgiApp)
%>

"""

import sys
from wsgiref.handlers import BaseCGIHandler


class ASPInputWrapper(object):

    def __init__(self, Request):
        self.stream = Request.BinaryRead
        size = Request.ServerVariables('CONTENT_LENGTH')
        self.remainder = self.size = int(size)

    def read(self, size=-1):
        if size lt; 0:
            size = self.remainder
        content, size = self.stream(size)
        self.remainder -= size
        return content

    def readline(self):
        output = []
        while True:
            # Use an internal buffer instead? Still have to check for \n
            char = self.read(1)
            if not char:
                break
            output.append(char)
            if char in ('\n', '\r'):
                break
        return ''.join(output)

    def readlines(self, hint=-1):
        lines = []
        while True:
            line = self.readline()
            if not line:
                break
            lines.append(line)
        return lines

    def __iter__(self):
        line = self.readline()
        while line:
            yield line
            # Notice this won't prefetch the next line; it only
            # gets called if the generator is resumed.
            line = self.readline()


class handler(BaseCGIHandler):

    def __init__(self, Application, Request, Response, buffering=True):
        # If you set buffering to False, you must not "Enable Buffering" in
        # the current Virtual Directory, NOR in any of its parent containers
        # (directory, site, or server). IIS 5 and 6 buffer by default.
        # See http://support.microsoft.com/default.aspx?scid=kb;en-us;Q306805
        # and http://www.aspfaq.com/show.asp?id=2262
        Response.Buffer = buffering

        env = {}
        for name in Request.ServerVariables:
            try:
                # names and values are both probably unicode. coerce them.
                env[str(name)] = str(Request.ServerVariables(name))
            except UnicodeEncodeError, x:
                # There's a potential problem lurking here, since some ASP
                # server var's which are required by WSGI may be high ASCII.
                x.args += ((u"Server Variable '%s'" % name),)
                raise x

        multiprocess = str(Application.Contents("multiprocess"))
        multithread = str(Application.Contents("multithread"))

        # You will probably need *some* form of rewriter to use ASP
        # with WSGI, since ASP requires one physical .asp file
        # per requestable-URL; so far, we support one:

        # Handle URL rewriting done by ISAPI_Rewrite Lite.
        # http://www.isapirewrite.com/
        # Note that PATH_TRANSLATED is also rewritten, but we
        # don't make any provision for unmunging that.
        old_path = env.get("HTTP_X_REWRITE_URL", None)
        if old_path:
            # Tear off any params.
            env["PATH_INFO"] = old_path.split("?")[0]

        # ASP puts the same values in SCRIPT_NAME and PATH_INFO,
        # for some odd reason. Empty one of them.
        env["SCRIPT_NAME"] = ""

        BaseCGIHandler.__init__(self,
                                stdin=ASPInputWrapper(Request),
                                stdout=None,
                                stderr=sys.stderr,
                                environ=env,
                                multiprocess=multiprocess,
                                multithread=multithread
                                )

        self.Response = Response
        self._write = Response.Write

    def _flush(self):
        self.Response.Flush()

    def send_headers(self):
        self.cleanup_headers()
        self.headers_sent = True
        for key, val in self.headers.items():
            self.Response.AddHeader(key, val)
