#! /usr/bin/env python

'''
Copyright (c) 2005, WukooPy Team (lihuimail@gmail.com)
All rights reserved.

Redistribution and use in source and binary forms, with or without modification, 
are permitted provided that the following conditions are met:

    * Redistributions of source code must retain the above copyright notice, 
      this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright notice, 
      this list of conditions and the following disclaimer in the documentation 
      and/or other materials provided with the distribution.
    * Neither the name of the WukooPy Team nor the names of its contributors 
      may be used to endorse or promote products derived from this software 
      without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE 
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE 
FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR 
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER 
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, 
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
'''

import thread
import sys, traceback, StringIO
import time
import urlparse
import cgitb
from quixote import get_request,get_publisher,get_session
from quixote.errors import PublishError, format_publish_error
from quixote import util
from quixote.config import Config
from quixote.http_response import HTTPResponse
from quixote.logger import DefaultLogger
from quixote.publish import Publisher
from quixote import publish
from quixote.session import NullSessionManager


from wukoopy.lib.singleton import Singleton



DEBUG=0


class Publisher(object):
    __metaclass__ = Singleton
    def __init__(self, root_directory, logger=None, session_manager=None,config=None, **kwargs):
        if publish._publisher is None:
            if not hasattr(self,'_request_dict'):
                setattr(self,'_request_dict',{})
            if DEBUG:
                print 'begin'
            if not hasattr(self,'config'):
                if DEBUG:
                    print 'init'
                if config is None:
                    self.config = Config(**kwargs)
                else:
                    if kwargs:
                        raise ValueError("cannot provide both 'config' object and config arguments")
                self.config = config
            if not hasattr(self,'logger'):
                if logger is None:
                    self.logger = DefaultLogger(error_log=self.config.error_log,
                        access_log=self.config.access_log,
                        error_email=self.config.error_email)
                else:
                    self.logger = logger
            if not hasattr(self,'session_manager'):
                if session_manager is not None:
                    self.session_manager = session_manager
                else:
                    self.session_manager = NullSessionManager()
            if not hasattr(self,'root_directory'):
                if not callable(getattr(root_directory, '_q_traverse')):
                    raise TypeError('Expected something with a _q_traverse method, got %r' %root_directory)
                self.root_directory = root_directory
        if publish._publisher is None:
            publish._publisher = self

    def _set_request(self, request):
        self._request_dict[thread.get_ident()] = request
    def _clear_request(self):
        try:
            del self._request_dict[thread.get_ident()]
        except KeyError:
            pass
    def get_request(self):
        return self._request_dict.get(thread.get_ident())

    #-------------------------------
    def set_session_manager(self, session_manager):
        self.session_manager = session_manager
    def log(self, msg):
        self.logger.log(msg)
    def parse_request(self, request):
        """Parse the request information waiting in 'request'.
        """
        request.process_inputs()
    def start_request(self):
        """Called at the start of each request.
        """
        self.session_manager.start_request()
    def finish_successful_request(self):
        """Called at the end of a successful request.
        """
        self.session_manager.finish_successful_request()
    def format_publish_error(self, exc):
        return format_publish_error(exc)
    def finish_interrupted_request(self, exc):
        """
        Called at the end of an interrupted request.  Requests are
        interrupted by raising a PublishError exception.  This method
        should return a string object which will be used as the result of
        the request.
        """
        if not self.config.display_exceptions and exc.private_msg:
            exc.private_msg = None # hide it
        request = get_request()
        request.response = HTTPResponse(status=exc.status_code)
        output = self.format_publish_error(exc)
        self.session_manager.finish_successful_request()
        return output
    def finish_failed_request(self):
        """
        Called at the end of an failed request.  Any exception (other
        than PublishError) causes a request to fail.  This method should
        return a string object which will be used as the result of the
        request.
        """
        # build new response to be safe
        request = get_request()
        original_response = request.response
        request.response = HTTPResponse()
        (exc_type, exc_value, tb) = sys.exc_info()
        error_summary = traceback.format_exception_only(exc_type, exc_value)
        error_summary = error_summary[0][0:-1] # de-listify and strip newline
        plain_error_msg = self._generate_plaintext_error(request,original_response,exc_type, exc_value,tb)
        if not self.config.display_exceptions:
            # DISPLAY_EXCEPTIONS is false, so return the most
            # secure (and cryptic) page.
            request.response.set_header("Content-Type", "text/html")
            user_error_msg = self._generate_internal_error(request)
        elif self.config.display_exceptions == 'html':
            # Generate a spiffy HTML display using cgitb
            request.response.set_header("Content-Type", "text/html")
            user_error_msg = self._generate_cgitb_error(request,original_response,exc_type, exc_value,tb)
        else:
            # Generate a plaintext page containing the traceback
            request.response.set_header("Content-Type", "text/plain")
            user_error_msg = plain_error_msg
        self.logger.log_internal_error(error_summary, plain_error_msg)
        request.response.set_status(500)
        self.session_manager.finish_failed_request()
        return user_error_msg
    def _generate_internal_error(self, request):
        admin = request.get_environ('SERVER_ADMIN',"<i>email address unknown</i>")
        return INTERNAL_ERROR_MESSAGE % admin
    def _generate_plaintext_error(self, request, original_response,exc_type, exc_value, tb):
        error_file = StringIO.StringIO()
        # format the traceback
        traceback.print_exception(exc_type, exc_value, tb, file=error_file)
        # include request and response dumps
        error_file.write('\n')
        error_file.write(request.dump())
        error_file.write('\n')
        return error_file.getvalue()
    def _generate_cgitb_error(self, request, original_response,exc_type, exc_value, tb):
        error_file = StringIO.StringIO()
        hook = cgitb.Hook(file=error_file)
        hook(exc_type, exc_value, tb)
        error_file.write('<h2>Original Request</h2>')
        error_file.write(str(util.dump_request(request)))
        error_file.write('<h2>Original Response</h2><pre>')
        original_response.write(error_file)
        error_file.write('</pre>')
        return error_file.getvalue()
    def try_publish(self, request):
        """(request : HTTPRequest) -> object

        The master method that does all the work for a single request.
        Exceptions are handled by the caller.
        """
        self.start_request()
        path = request.get_environ('PATH_INFO', '')
        assert path[:1] == '/'
        # split path into components
        path = path[1:].split('/')
        output = self.root_directory._q_traverse(path)
        # The callable ran OK, commit any changes to the session
        self.finish_successful_request()
        return output
    def filter_output(self, request, output):
        """Hook for post processing the output.  Subclasses may wish to
        override (e.g. check HTML syntax).
        """
        return output
    def process_request(self, request):
        """(request : HTTPRequest) -> HTTPResponse

        Process a single request, given an HTTPRequest object.  The
        try_publish() method will be called to do the work and
        exceptions will be handled here.
        """
        self._set_request(request)
        start_time = time.time()
        try:
            self.parse_request(request)
            output = self.try_publish(request)
        except PublishError, exc:
            # Exit the publishing loop and return a result right away.
            output = self.finish_interrupted_request(exc)
        except:
            # Some other exception, generate error messages to the logs, etc.
            output = self.finish_failed_request()
        output = self.filter_output(request, output)
        self.logger.log_request(request, start_time)
        if output:
            if self.config.compress_pages and request.get_encoding(["gzip"]):
                compress = True
            else:
                compress = False
            request.response.set_body(output, compress)
        self._clear_request()
        return request.response






def get_url():
    result=get_request().get_url()
    if get_request().get_query():
        result+='?'+get_request().get_query()
    return result

def get_clientip():
    return get_request().environ.get('REMOTE_ADDR','')


def get_config():
    return get_publisher().config

def get_form(v):
    return get_request().get_fields(v)

def get_post():
    request=get_request()
    if request.get_method() != 'POST':
        request.response.set_status(405, "Only the POST method is accepted")
        return "XML-RPC handlers only accept the POST method."
    length = int(request.environ['CONTENT_LENGTH'])
    result = request.stdin.read(length)
    return result


def get_user(t,default=None):
    f=get_session().user
    if hasattr(f,t):
        result=getattr(f,t)
    else:
        result=default
    return result
