"""
pyrss2 - a plugin for generating a RSS2.0 feed with ?flav=rss2.

Releaed under a Creative Commons Atribution ShareAlike license

  http://creativecommons.org/licenses/by-sa/1.0/

DESCRIPTION:

Adds additional template variables for use with templates. A default
set of templates are supplied. The template variables use a direct
mapping of names with the XML fields.

By default, this plugin looks for .rss2 flavour files in the configured
datadir and substitutes a particular set of variables not available
unless this plugin is called. Some of the variables just copy
the appropriate information from config.py.

CONFIGURATION:

To configure this plugin, please edit the appropriate variables below.

TEMPLATE VARIABLES:

@type rss_title: string
@param rss_title: Title of the Rss feed. Grabs ['blog_title'].

@type rss_author: string
@param rss_author: Author information. Grabs ['blog_author']. 

@type rss_description: string
@param rss_description: Rss feed description. Grabs ['blog_description'].

@type rss_language: string
@param rss_language: Rss feed language. Grabs ['blog_language']

@type rss_lastbuilddate: string
@param rss_lastbuilddate: Date of the newest entry in the feed.

@type rss_length_item_desc: int
@param rss_length_item_desc: How many words of each entry to include
	in the description the feed. The default value is 0. Values < 0
	are interpreted as all words in the entry.

@type rss_item_desc: string
@param rss_item_desc: template variable which is exchanged with the 
	first rss_length_item_desc words of each entry.

@type rss_copyritgt: string
@param rss_copyright: User supplied copyright string for the Rss feed.

@type rss_docs: string
@param rss_docs: The URI of the RSS2.0 specification.

@type rss_generator: string
@param rss_generator: Verision of pyblosxom used + version of plugin.

"""

# User editable variables begin

# Set a copyright notice here.
rss_copyright = "(c) 2003 Rune Morling - under a creative commons Attribute ShareAlike license"

# Set how many words of each entry is to be shown in the rss feed.
# To show the whole entry, insert a negative number (i.e. -1). 
# If you use a number greater than the length of the entry,
# the effect will be same as with -1. Defaults to 0
rss_entry_length_desc = 20

# Let aggregators know how long your rss-feed should be cached in minutes
# before a new request to it should be made. Expected to be a positive 
# integer. Negative numbers are converted to 0. Defaults to 60 minutes
rss_ttl = 60

# User editable variables end
# -- Nothing to set below this line --

__author__ = "Rune Morling - pyblosxom at u7 dot dk"
__version__ = "$Id: pyrss2.py,v 0.2 2003/11/03 17:57:13 ermo Exp $"

# link to RSS2 spec
rss_docs = "http://blogs.law.harvard.edu/tech/rss"

from Pyblosxom import tools
from string import split, join
from math import floor

def get_newest_entry(request_data):
	"""
	Get the date of the newest entry in the blog.
	I think this might involve just picking the first FileEntry object
	in the request assuming the list is sorted already...
	
	@type request_data: unpacked Request object
	@param request_data: Contains the sorted entry list we're interested in.

	@rtype: string
	@return: The date of the newest entry in the feed
	"""
	# get the entries in a sorted list
	entries = request_data["entry_list"]
	#on the assumption that the newest entry is first	
	newest_entry = entries[0] 

	#generate date string. The GMT part is a stupid hack. *FIXME*
	year = newest_entry.getMetadata("yr", "this year")
	month = newest_entry.getMetadata("mo", "this month")
	date = newest_entry.getMetadata("da", "today")
	time = newest_entry.getMetadata("ti", "now")
	rss_lastbuilddate = date + " " + month + " " + year + " " + time + ":00 GMT"

	return rss_lastbuilddate

def get_version():
	"""
	@rtype: string
	@return: A string containing the pyblosxom and pyrss2 version.
	"""
	# stupid hack - *FIXME* 
	return "PyBlosxom v0.8.1, pyRss2 0.2"

def get_ttl(ttl):
	"""
	Converts negative numbers to zero. Also floors decimal numbers.

	@type ttl: int
	@param rss_ttl: Hopefully a nice positive integer.

	@type: int returns a whole number >= 0. 
	@returns: An integer >= 0. Defaults to 60 if someone goofed...

	"""
	i = 60 # sane, default value if everything else fails!
		   # also, this would be a hint to a type checking compiler ...
		# *FIXME* I should check if this is a number?
	try:
		i = int(ttl)
		if i < 0:
			i = 0
	except:
		pass #We've already set a sane default

	return i

def get_rss_entry_length_desc(entry):
	"""
	Takes a string (an entry, really) and optionally discards or truncates
	it based on the value of rss_entry_length_desc. Defaults to the empty
	string (in case someone made a mistake)

	@type entry: string
	@param entry: An entry from the blog.

	@rtype: string
	@returns: The (optionally) truncated entry. Defaults to the empty string.
	"""
	# sane default in case we're passed something silly
	rss_desc = ""

	try:
		# Are we dealing with a number?
		entry_length = int(rss_entry_length_desc)
		if entry_length == 0:
			pass
		elif entry_length < 0:
			# strip html tags.
			strip = tools.Stripper()
			strip.handle_data(entry)
#			rss_desc = "<![CDATA[ " + strip.gettext() + " ]]>"
			rss_desc = strip.gettext()
		else:
			# truncate the entry to rss_length_item_desc words or less
			# this part is stupid - it doesn't check for closing tags and such.
			# strip html tags
			strip = tools.Stripper()
			strip.handle_data(entry)
			words = split( strip.gettext() )

			desc = []

			# Make sure we don't run out of indices
			max_index = len(words)
			if max_index < entry_length:
				entry_length = max_index
	
			for i in range(entry_length):
				desc.append(words[i])
	
			# Use CDATA to escape odd syntax in description
#			rss_desc = "<![CDATA[ " + join(desc) + " ...]]>"
			rss_desc = join(desc)
	except:
		pass

	return rss_desc

# make pyblosxom recognize the extra header variables so that we can
# populate the rss2 flavour specific template vars
def cb_prepare(args):
	request = args["request"]
	data = request.getData()
	config = request.getConfiguration()

	# set variables in the request object and send the request
	# object on for further processing
	data["rss_title"] 		=	config["blog_title"]
	data["rss_link"] 		=	config["base_url"]
	data["rss_author"] 		= 	config["blog_author"]
	data["rss_description"] = 	config["blog_description"]
	data["rss_language"] 	=   config["blog_language"]
	data["rss_lastbuilddate"] = get_newest_entry(data)
	data["rss_generator"] 	=   get_version()
	data["rss_ttl"]			=   get_ttl(rss_ttl)
	data["rss_copyright"]	= 	rss_copyright
	data["rss_docs"]		= 	rss_docs

# Substitute $rss_entry_desc in entry.rss2 template
# Is it possible to check what kind of ?flav we are using,
# and make the truncation if and only if this is a ?flav=rss2 call?
# Because right now, this plugin is run regardless of flavour ...)
def cb_story(args):

	entry = args["entry"]
	#truncate entry?
	entry_replace = get_rss_entry_length_desc(entry["body"])
	entry["rss_entry_desc"] = entry_replace
