from autumn.db.query import Query,escape
from autumn.db.connection import autumn_db, Database

import re

def lower_name(class_name):
    """
    >>>lower_name("UserCount")
    'user_count'

    >>>lower_name("user_count") 
    'user_count'
    """
    return re.sub('(((?<=[a-z])[A-Z])|([A-Z](?![A-Z]|$)))', '_\\1', class_name).lstrip("_").lower()

class ModelCache(object):
    models = {}

    def add(self, model):
        self.models[model.__name__] = model

    def get(self, model_name):
        return self.models[model_name]

cache = ModelCache()

class ModelBase(type):
    '''
    Metaclass for Model
    
    Sets up default table name and primary key
    Adds fields from table as attributes
    Creates ValidatorChains as necessary
    
    '''
    def __new__(cls, name, bases, attrs):
        if name == 'Model':
            return super(ModelBase, cls).__new__(cls, name, bases, attrs)

        new_class = type.__new__(cls, name, bases, attrs)

        if not getattr(new_class, 'Meta', None):
            class Empty:
                pass
            new_class.Meta = Empty

        if not getattr(new_class.Meta, 'table', None):
            new_class.Meta.table = lower_name(name)
        new_class.Meta.table_safe = escape(new_class.Meta.table)

        # Assume id is the default 
        if not getattr(new_class.Meta, 'pk', None):
            new_class.Meta.pk = 'id'

        # See cursor.description
        # http://www.python.org/dev/peps/pep-0249/
        if not hasattr(new_class, "db"):
            new_class.db = autumn_db
        db = new_class.db

        q = db.conn.connection.cursor()
        q.execute('SELECT * FROM %s LIMIT 0' % new_class.Meta.table_safe)
        db.conn.connection.commit()

        new_class._fields = [f[0] for f in q.description]

        cache.add(new_class)
        return new_class

class Model(object):
    '''
    Allows for automatic attributes based on table columns.
    
    Syntax::
    
        from autumn.model import Model
        class MyModel(Model):
            class Meta:
                # If field is blank, this sets a default value on save
                class default:
                    field = 1
            
                # Table name is lower-case model name by default
                # Or we can set the table name
                table = 'mytable'
        
        # Create new instance using args based on the order of columns
        m = MyModel(1, 'A string')
        
        # Or using kwargs
        m = MyModel(field=1, text='A string')
        
        # Saving inserts into the database (assuming it validates [see below])
        m.save()
        
        # Updating attributes
        m.field = 123
        
        # Updates database record
        m.save()
        
        # Deleting removes from the database 
        m.delete()
        
        m = MyModel(field=0)
        
        m.save()
        
        # Retrieval is simple using Model.get
        # Returns a Query object that can be sliced
        MyModel.filter()
        
        # Returns a MyModel object with an id of 7
        m = MyModel.one(7)
        
        # Limits the query results using SQL's LIMIT clause
        # Returns a list of MyModel objects
        m = MyModel.filter()[:5]   # LIMIT 0, 5
        m = MyModel.filter()[10:15] # LIMIT 10, 5
        
        # We can get all objects by slicing, using list, or iterating
        m = MyModel.get()[:]
        m = list(MyModel.filter())
        for m in MyModel.filter():
            # do something here...
            
        # We can filter our Query
        m = MyModel.filter(field=1)
        m = m.filter(another_field=2)
        
        # This is the same as
        m = MyModel.filter(field=1, another_field=2)
        
        # Set the order by clause
        m = MyModel.filter(field=1).order_by('field', 'DESC')
        # Removing the second argument defaults the order to ASC
        
    '''
    __metaclass__ = ModelBase

    debug = False

    def __init__(self, *args, **kwargs):
        'Allows setting of fields using kwargs'
        self.__dict__[self.Meta.pk] = None
        self._new_record = True
        for i, arg in enumerate(args):
            self.__dict__[self._fields[i]] = arg
        for i in self._fields[len(args):]:
            self.__dict__[i] = kwargs.get(i)
        self.__dict__["_changed"] = set()

    def __setattr__(self, name, value):
        'Records when fields have changed'
        dc=self.__dict__
        if name[0]!="_":
            fields=self._fields
            if name in fields:
                if dc[name]!=value:
                    self._changed.add(name)
        dc[name] = value

    def _get_pk(self):
        'Returns value of primary key'
        return getattr(self, self.Meta.pk)

    def _get_pk(self):
        'Sets the current value of the primary key'
        return getattr(self, self.Meta.pk, None)

    def _set_pk(self, value):
        'Sets the primary key'
        return setattr(self, self.Meta.pk, value)

    def _update(self):
        if not self._changed:return
        'Uses SQL UPDATE to update record'
        query = 'UPDATE %s SET ' % self.Meta.table_safe
        query += ','.join(['%s=%s' % (escape(f), self.db.conn.placeholder) for f in self._changed])
        query += ' WHERE %s=%s ' % (escape(self.Meta.pk), self.db.conn.placeholder)

        values = [getattr(self, f) for f in self._changed]
        values.append(self._get_pk())

        cursor = Query.raw_sql(query, values, self.db)

    def _new_save(self):
        'Uses SQL INSERT to create new record'
        # if pk field is set, we want to insert it too
        # if pk field is None, we want to auto-create it from lastrowid
        auto_pk = 1 and (self._get_pk() is None) or 0
        fields = [
            f for f in self._fields
            if f != self.Meta.pk or not auto_pk
        ]

        used_fields=[]
        values=[]
        for i in fields:
            v = getattr(self, i, None)
            if v is not None:
                used_fields.append(escape(i))
                values.append(v)
        query = 'INSERT INTO %s (%s) VALUES (%s)' % (self.Meta.table_safe,
                ', '.join(used_fields),
                ', '.join([self.db.conn.placeholder] * len(used_fields))
        )
        cursor = Query.raw_sql(query, values, self.db)


        if self._get_pk() is None:
            self._set_pk(cursor.lastrowid)
        return True

    def _set_default(self):
        if hasattr(self.Meta, 'default'):
            default = self.Meta.default
            i = default()

            for k, v in default.__dict__.iteritems():
                if k[0] != '_' :
                    if getattr(self,k,None) is None:
                        if callable(v):
                            v = getattr(i,k)()
                        setattr(self,k,v)
    @classmethod
    def raw_sql(cls,query,*args):
        result = Query.raw_sql(query, args, cls.db)
        return result
        
    def delete(self):
        'Deletes record from database'
        query = 'DELETE FROM %s WHERE `%s` = %s' % (self.Meta.table_safe, self.Meta.pk, self.db.conn.placeholder)
        values = [getattr(self, self.Meta.pk)]
        Query.raw_sql(query, values, self.db)

    def save(self):
        if self._new_record:
            self._set_default()
            self._new_save()
            self._new_record = False
        else:
            self._update()
        return self

    @classmethod
    def filter(cls,*args, **kwargs):
        'Returns Query object'
        return Query(
            model=cls,
            args=args,
            conditions=kwargs
        )

    @classmethod
    def one(cls, __obj_pk=None, **kwargs):
        if __obj_pk is not None:
            kwargs={
                cls.Meta.pk: __obj_pk
            }
        q = Query(model=cls, conditions=kwargs)
        q = q.execute_query()
        q = q.fetchone()
        if q:
            obj = cls(*q)
            obj.__dict__['_new_record'] = False
            return obj


    def save(self):
        if self._new_record:
            self._set_default()
            self._new_save()
            self._new_record = False
        else:
            self._update()
        return self

    def replace_into(self):
        used_fields = []
        values = []
        for i in self._fields:
            v = getattr(self, i, None)
            if v is not None:
                used_fields.append(escape(i))
                values.append(v)
        query = 'REPLACE INTO %s (%s) VALUES (%s)' % (
                self.Meta.table_safe,
                ', '.join(used_fields),
                ', '.join([self.db.conn.placeholder] * len(used_fields))
        )
        cursor = Query.raw_sql(query, values, self.db)

    @classmethod
    def get_or_create(cls,*args,**kwds):
        ins = cls.one(*args,**kwds)
        if not ins:
            ins = cls(*args,**kwds)
            ins.save()
        return ins
