﻿# vim: set fileencoding=utf-8 :
from django.db import models
from django.core.validators import ValidationError

def make_radio_input(text, **attrs):
    return '<label><input type="radio" %s /> %s </label>' %\
            (' '.join('%s="%s"'%(str(k), str(v)) for k, v in attrs.items()),
             text)

class ChooseQuestion(models.Model):
    question = models.TextField('题目', maxlength=500)
    choice1 = models.CharField('选项A', maxlength=100)
    choice2 = models.CharField('选项B', maxlength=100)
    choice3 = models.CharField('选项C', maxlength=100)
    choice4 = models.CharField('选项D', maxlength=100)
    author = models.CharField('作者', maxlength=20, null=True, blank=True)

    def __str__(self):
        return self.question

    def to_html(self, id, value=None):
        if not value:
            value = 'A'
        values = ['A', 'B', 'C', 'D']
        radios = [(values[i]==value) and\
                    make_radio_input(choice, name=id, checked="checked", value=values[i]) or\
                    make_radio_input(choice, name=id, value=values[i])\
                for i, choice in enumerate((self.choice1, self.choice2,
                                        self.choice3, self.choice4))]
        choices = '<ul class="choices">\n%s\n</ul>' %\
                '\n'.join(['<li>%s</li>' % w for w in radios])

        return '<br/>'.join([self.question, choices])

    class Meta:
        verbose_name = verbose_name_plural = '选择题'

    class Admin:
        list_display = ('question', 'choice1', 'choice2', 'choice3',
                    'choice4', 'author')

class Blanks(object):
    def __init__(self, id, test=False, value=None):
        self.id = id
        self.test = test
        self.value = value
        self.count = 0
    def __getitem__(self, name):
        size = int(name[5:])
        if self.test:
            result = '_'*size
        elif not self.value:
            result = '<input type="text" name="%s" size="%d"/>' %\
                    (self.id, size)
        else:
            result = '<input type="text" name="%s" size="%d" value="%s"/>' %\
                    (self.id, size, self.value[self.count])

        self.count += 1
        return result

def isQuestion(field_data, all_data):
    try:
        field_data % Blanks('a', True)
    except Exception, e:
        raise ValidationError('题目的格式有错！%s:%s'%(e.__class__.__name__, e))

class BlankQuestion(models.Model):
    question = models.TextField('题目', maxlength=500,
            validator_list=[isQuestion])
    author = models.CharField('作者', maxlength=20, null=True, blank=True)

    def __str__(self):
        return self.question

    def to_html(self, id, value=None):
        if value:
            assert isinstance(value, list)
        blanks = Blanks(id, value=value)
        return self.question % blanks

    class Meta:
        verbose_name = verbose_name_plural = '填空题'

    class Admin:
        list_display = ('question', 'author')

class AnswerQuestion(models.Model):
    question = models.TextField('题目', maxlength=500)
    author = models.CharField('作者', maxlength=20, null=True, blank=True)

    def __str__(self):
        return self.question

    def to_html(self, id, value=None):
        if not value:
            return '<br/>'.join([self.question, 
                '<textarea rows="10" cols="40" name="%s"></textarea>'%id])
        else:
            return '<br/>'.join([self.question,
                '<textarea rows="10" cols="40" name="%s">%s</textarea>'%(id, value)])

    class Meta:
        verbose_name = verbose_name_plural = '问答题'

    class Admin:
        list_display = ('question', 'author')

class Paper(models.Model):
    title = models.CharField('试卷标题', maxlength=200)
    cqs = models.ManyToManyField(ChooseQuestion, verbose_name='选择题',
            blank=True, filter_interface=models.HORIZONTAL)
    bqs = models.ManyToManyField(BlankQuestion, verbose_name='填空题',
            blank=True, filter_interface=models.HORIZONTAL)
    aqs = models.ManyToManyField(AnswerQuestion, verbose_name='问答题',
            blank=True, filter_interface=models.HORIZONTAL)
    author = models.CharField('作者', maxlength=20, null=True, blank=True)

    def __str__(self):
        return self.title

    class Meta:
        verbose_name = verbose_name_plural = '试卷'

    class Admin:
        pass
