#!/usr/bin/env python
# -*- coding: utf-8 -*-
# report bugs to: riverfor@gmail.com
"""
test module of compass.protocol.gnspmsg
"""
from framework import run, unittest
from compass.messages import gnspmsg

def packmsg(name, items):
    m = gnspmsg.GNSPMessage()
    m.setMsgName(name)
    for f,v in items:
        m.body.setField(f, v)
    return m.packed()

def unpackmsg(packet):
    m = gnspmsg.GNSPMessage()
    m.loadMessage(packet)
    return m

### test parent class
class ConnectMessageTestCase(unittest.TestCase):
    msgname = 'connect'
    items = (('service_id', 'sys001'),
             ('auth_source', 'xyb4567890123456'),
             ('version', 0x0100),
             ('time_stamp', '200411191139420000'),
            )
    packet = '\x00\x00\x00^\x00\x00\x00\x01\x00\x00\x00\x00sys001\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00xyb4567890123456\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x01\x00\x00200411191139420000\x00\x00\x00\x00'

    def testMessage(self):
        plen = len(self.packet)
        # test package a message
        m = packmsg(self.msgname, self.items)
        self.assertEqual(m, self.packet)
        # test unpack
        n = unpackmsg(self.packet)
        for f,v in self.items:
            self.assertEqual(n.body[f], v)
        self.assertEqual(n.head.length, plen)
        self.assertEqual(n.packed(), self.packet)
        self.assertEqual(n.msgname, self.msgname)
### test the ConnectRespMessage
class ConnectRespMessageTestCase(ConnectMessageTestCase):
    msgname = 'connect_resp'
    items = (('status', 0xfedc),
             ('version', 0x0001),
            )
    packet = '\x00\x00\x00\x14\xff\x00\x00\x01\x00\x00\x00\x00\xdc\xfe\x00\x00\x01\x00\x00\x00'

### test the DisconnectMessage
class DisconnectMessageTestCase(ConnectMessageTestCase):
    msgname = 'disconnect'
    items = ()
    packet = '\x00\x00\x00\x0c\x00\x00\x00\x02\x00\x00\x00\x00'

### test the DisconnectRespMessage
class DisconnectRespMessageTestCase(ConnectMessageTestCase):
    msgname = 'disconnect_resp'
    items = ()
    packet = '\x00\x00\x00\x0c\xff\x00\x00\x02\x00\x00\x00\x00'

### test ApplicationMessage
class AddApplicationMessageTestCase(ConnectMessageTestCase):
    msgname = 'add_application'
    items = (('application_id', 'appid'),
             ('desc', 'desc'),
            )
    packet = '\x00\x00\x00\\\x00\x00\x00\x10\x00\x00\x00\x00appid\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00desc\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00'

### test AddApplicationRespMessage
class AddApplicationRespMessageTestCase(ConnectMessageTestCase):
    msgname = 'add_application_resp'
    items = (('status', 0x1234),)
    packet = '\x00\x00\x00\x10\xff\x00\x00\x10\x00\x00\x00\x004\x12\x00\x00'

### test DelApplicationMessageTestCase
class DelApplicationMessageTestCase(ConnectMessageTestCase):
    msgname = 'del_application'
    items = (('application_id', 'appid'),)
    packet = '\x00\x00\x00\x1c\x00\x00\x00\x11\x00\x00\x00\x00appid\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00'

### test DelApplicationRespMessage
class DelApplicationRespMessageTestCase(ConnectMessageTestCase):
    msgname = 'del_application_resp'
    items = (('status', 0x1234),)
    packet = '\x00\x00\x00\x10\xff\x00\x00\x11\x00\x00\x00\x004\x12\x00\x00'

### test ModifyApplicationMessage
class ModifyApplicationMessageTestCase(ConnectMessageTestCase):
    msgname = 'modify_application'
    items = (('application_id', 'appid'),
             ('desc', 'desc'),
            )
    packet = '\x00\x00\x00\\\x00\x00\x00\x12\x00\x00\x00\x00appid\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00desc\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00'

### test ModifyApplicationRespMessage
class ModifyApplicationRespMessageTestCase(ConnectMessageTestCase):
    msgname = 'modify_application_resp'
    items = (('status', 0x1234),)
    packet = '\x00\x00\x00\x10\xff\x00\x00\x12\x00\x00\x00\x004\x12\x00\x00'

### test GetApplicationMessage
class GetApplicationMessageTestCase(ConnectMessageTestCase):
    msgname = 'get_application'
    items = (('application_id', 'appid'),)
    packet = '\x00\x00\x00\x1c\x00\x00\x00\x13\x00\x00\x00\x00appid\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00'

#### base test case class for list resp messages
class GetApplicationRespMessageTestCase(unittest.TestCase):
    msgname = 'get_application_resp'
    items = (('status', 0x1234),
             ('application_num', 0x0002),
            )
    resplist = ({'application_id': 'app01',
                 'desc': "app01's desc",
                },
                {'application_id': 'app02',
                 'desc': "app02's desc"
                },
               )
    packet = "\x00\x00\x00\xb4\xff\x00\x00\x13\x00\x00\x00\x004\x12\x00\x00\x02\x00\x00\x00app01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00app01's desc\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00app02\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00app02's desc\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"

    def testMessage(self):
        m = gnspmsg.GNSPMessage()
        m.setMsgName(self.msgname)
        for f,v in self.items:
            m.body.setField(f, v)
        m.addlist(self.resplist)
        self.assertEquals(m.packed(), self.packet)

        m = gnspmsg.GNSPMessage()
        m.loadMessage(self.packet)
        self.assertEquals(m.msgname, self.msgname)
        for f,v in self.items:
            self.assertEqual(m.body[f], v)
        for i, item in enumerate(m.getlist()):
            self.assertEqual(item, self.resplist[i])
        self.assertEqual(m.packed(), self.packet)

### test GetApplicationRespMessage
class GetApplicationRespMessage0TestCase(GetApplicationRespMessageTestCase):
    msgname = 'get_application_resp'
    items = (('status', 0x1234),
             ('application_num', 0x0000),
            )
    resplist = ()
    packet = '\x00\x00\x00\x14\xff\x00\x00\x13\x00\x00\x00\x004\x12\x00\x00\x00\x00\x00\x00'

class GetApplicationRespMessage1TestCase(GetApplicationRespMessageTestCase):
    msgname = 'get_application_resp'
    items = (('status', 0x1234),
             ('application_num', 0x0001),
            )
    resplist = ({'application_id': 'app01',
                 'desc': "app01's desc",
                },
               )
    packet = "\x00\x00\x00d\xff\x00\x00\x13\x00\x00\x00\x004\x12\x00\x00\x01\x00\x00\x00app01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00app01's desc\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"

### test AddServiceGroupMesage
class AddServiceGroupMessageTestCase(ConnectMessageTestCase):
    msgname = 'add_servicegroup'
    items = (('servicegroup_id', 'sgid'),
             ('desc', 'a service group'),
            )
    packet = '\x00\x00\x00\\\x00\x00\x00\x14\x00\x00\x00\x00sgid\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00a service group\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00'

### test AddServiceGroupRespMessage
class AddServiceGroupRespMessageTestCase(ConnectMessageTestCase):
    msgname = 'add_servicegroup_resp'
    items = (('status', 0x1234),)
    packet = '\x00\x00\x00\x10\xff\x00\x00\x14\x00\x00\x00\x004\x12\x00\x00'

### test ModifyserviceGroupMessage
class ModifyServiceGroupMessageTestCase(ConnectMessageTestCase):
    msgname = 'modify_servicegroup'
    items = (('servicegroup_id', 'sgid'),
             ('desc', 'a service group'),
            )
    packet = '\x00\x00\x00\\\x00\x00\x00\x15\x00\x00\x00\x00sgid\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00a service group\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00'

### test ModifyServiceGroupRespMessage
class ModifyServiceGroupRespMessageTestCase(ConnectMessageTestCase):
    msgname = 'modify_servicegroup_resp'
    items = (('status', 0x1234),)
    packet = '\x00\x00\x00\x10\xff\x00\x00\x15\x00\x00\x00\x004\x12\x00\x00'

### test DelServiceGroupMessage
class DelServiceGroupMessageTestCase(ConnectMessageTestCase):
    msgname = 'del_servicegroup'
    items = (('servicegroup_id', 'sgid'),)
    packet = '\x00\x00\x00\x1c\x00\x00\x00\x16\x00\x00\x00\x00sgid\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00'

### test DelServiceGroupRespMessage
class DelServiceGroupRespMessageTestCase(ConnectMessageTestCase):
    msgname = 'del_servicegroup_resp'
    items = (('status', 0x1234),)
    packet = '\x00\x00\x00\x10\xff\x00\x00\x16\x00\x00\x00\x004\x12\x00\x00'

### test GetServiceGroupMessage
class GetServiceGroupMessageTestCase(ConnectMessageTestCase):
    msgname = 'get_servicegroup'
    items = (('servicegroup_id', 'sgid'),)
    packet = '\x00\x00\x00\x1c\x00\x00\x00\x17\x00\x00\x00\x00sgid\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00'

### test GetServiceGRoupRespMessage
class GetServiceGroupRespMessageTestCase(GetApplicationRespMessageTestCase):
    msgname = 'get_servicegroup_resp'
    items = (('status', 0x1234),
             ('servicegroup_num', 0x0002),
             ('list_num', 0x0002),
            )
    resplist = ({'servicegroup_id': 'sg01',
                 'desc': "sg01's desc",
                },
                {'servicegroup_id': 'sg02',
                 'desc': "sg02's desc"
                },
               )
    packet = "\x00\x00\x00\xb8\xff\x00\x00\x17\x00\x00\x00\x004\x12\x00\x00\x02\x00\x00\x00\x02\x00\x00\x00sg01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00sg01's desc\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00sg02\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00sg02's desc\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"

### test AddServiceMessage
class AddServiceMessageTestCase(ConnectMessageTestCase):
    msgname = 'add_service'
    items = (('service_id', 'serviceid'),
             ('desc', 'a service'),
             ('password', 'password'),
             ('host', 'hostname'),
             ('port', 0x7890),
             ('domain', 0x0001),
             ('tactics', 0x0002),
             ('status', 'A'),
            )
    packet = '\x00\x00\x00\x9d\x00\x00\x00\x18\x00\x00\x00\x00serviceid\x00\x00\x00\x00\x00\x00\x00a service\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00password\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00hostname\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x90x\x00\x00\x01\x00\x00\x00\x02\x00\x00\x00A'

### test AddServiceRespMessage
class AddServiceRespMessageTestCase(ConnectMessageTestCase):
    msgname = 'add_service_resp'
    items = (('status', 0x1234),)
    packet = '\x00\x00\x00\x10\xff\x00\x00\x18\x00\x00\x00\x004\x12\x00\x00'

### test ModifyServiceMessage
class ModifyServiceMessageTestCase(ConnectMessageTestCase):
    msgname = 'modify_service'
    items = (('service_id', 'serviceid'),
             ('mask', 0x000f),
             ('desc', 'a service'),
             ('password', 'password'),
             ('host', 'hostname'),
             ('port', 0x7890),
             ('domain', 0x0001),
             ('tactics', 0x0002),
             ('status', 'A'),
            )
    packet = '\x00\x00\x00\xa1\x00\x00\x00\x19\x00\x00\x00\x00serviceid\x00\x00\x00\x00\x00\x00\x00\x0f\x00\x00\x00a service\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00password\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00hostname\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x90x\x00\x00\x01\x00\x00\x00\x02\x00\x00\x00A'

### test ModifyServiceRespMessage
class ModifyServiceRespMessageTestCase(ConnectMessageTestCase):
    msgname = 'modify_service_resp'
    items = (('status', 0x1234),)
    packet = '\x00\x00\x00\x10\xff\x00\x00\x19\x00\x00\x00\x004\x12\x00\x00'

### test DelServiceMessage
class DelServiceMessageTestCase(ConnectMessageTestCase):
    msgname = 'del_service'
    items = (('service_id', 'serviceid'),)
    packet = '\x00\x00\x00\x1c\x00\x00\x00\x1a\x00\x00\x00\x00serviceid\x00\x00\x00\x00\x00\x00\x00'

### test DelServiceRespMessage
class DelServiceRespMessageTestCase(ConnectMessageTestCase):
    msgname = 'del_service_resp'
    items = (('status', 0x1234),)
    packet = '\x00\x00\x00\x10\xff\x00\x00\x1a\x00\x00\x00\x004\x12\x00\x00'

### test GetServiceMessage
class GetServiceMessageTestCase(ConnectMessageTestCase):
    msgname = 'get_service'
    items = (('service_id', 'serviceid'),)
    packet = '\x00\x00\x00\x1c\x00\x00\x00\x1b\x00\x00\x00\x00serviceid\x00\x00\x00\x00\x00\x00\x00'

### test GetServiceRespMessage
class GetServiceRespMessageTestCase(GetApplicationRespMessageTestCase):
    msgname = 'get_service_resp'
    items = (('status', 0x1234),
             ('service_num', 0x0002),
             ('list_num', 0x0002),
            )
    resplist = ({'service_id': 's01',
                 'desc': "s01's desc",
                 'host': 'host01',
                 'port': 0x1234,
                 'listen_host': 'host01',
                 'listen_port': 0x2345,
                 'domain': 0x0001,
                 'tactics': 0x0002,
                 'status': 'A',
                },
                {'service_id': 's02',
                 'desc': "s02's desc",
                 'host': 'host02',
                 'port': 0x1234,
                 'listen_host': 'host02',
                 'listen_port': 0x2345,
                 'domain': 0x0002,
                 'tactics': 0x0002,
                 'status': 'D',
                },
               )
    packet = "\x00\x00\x01Z\xff\x00\x00\x1b\x00\x00\x00\x004\x12\x00\x00\x02\x00\x00\x00\x02\x00\x00\x00s01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00s01's desc\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00host01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x004\x12\x00\x00host01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00E#\x00\x00\x01\x00\x00\x00\x02\x00\x00\x00As02\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00s02's desc\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00host02\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x004\x12\x00\x00host02\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00E#\x00\x00\x02\x00\x00\x00\x02\x00\x00\x00D"


### test AddDependMessage
class AddDependMessageTestCase(ConnectMessageTestCase):
    msgname = 'add_depend'
    items = (('from_id', 'from serviceid'),
             ('to_id', 'to serviceid'),
            )
    packet = '\x00\x00\x00,\x00\x00\x00\x1c\x00\x00\x00\x00from serviceid\x00\x00to serviceid\x00\x00\x00\x00'

### test AddDependRespMessage
class AddDependRespMessageTestCase(ConnectMessageTestCase):
    msgname = 'add_depend_resp'
    items = (('status', 0x1234),)
    packet = '\x00\x00\x00\x10\xff\x00\x00\x1c\x00\x00\x00\x004\x12\x00\x00'

### test DelDependMessage
class DelDependMessageTestCase(ConnectMessageTestCase):
    msgname = 'del_depend'
    items = (('from_id', 'from serviceid'),
             ('to_id', 'to serviceid'),
            )
    packet = '\x00\x00\x00,\x00\x00\x00\x1d\x00\x00\x00\x00from serviceid\x00\x00to serviceid\x00\x00\x00\x00'

### test DelDependRespMessage
class DelDependRespMessageTestCase(ConnectMessageTestCase):
    msgname = 'del_depend_resp'
    items = (('status', 0x1234),)
    packet = '\x00\x00\x00\x10\xff\x00\x00\x1d\x00\x00\x00\x004\x12\x00\x00'

### test GetDependMessage
class GetDependMessageTestCase(ConnectMessageTestCase):
    msgname = 'get_depend'
    items = (('from_id', 'from serviceid'),
             ('to_id', 'to serviceid'),
            )
    packet = '\x00\x00\x00,\x00\x00\x00\x1e\x00\x00\x00\x00from serviceid\x00\x00to serviceid\x00\x00\x00\x00'

### test GetDependRespMessage
class GetDependRespMessageTestCase(GetApplicationRespMessageTestCase):
    msgname = 'get_depend_resp'
    items = (('status', 0x1234),
             ('depend_num', 0x0001),
            )
    resplist = ({'from_id': 'sg01',
                 'to_id': 'sg02',
                },
               )
    packet = '\x00\x00\x004\xff\x00\x00\x1e\x00\x00\x00\x004\x12\x00\x00\x01\x00\x00\x00sg01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00sg02\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00'

### test ServiceLoginMessage
class ServiceLoginMessageTestCase(ConnectMessageTestCase):
    msgname = 'service_login'
    items = (('service_id', 'serviceid'),
             ('listen_host', 'hostname'),
             ('listen_port', 0x7890),
             ('status', 'A'),
            )
    packet = '\x00\x00\x00A\x00\x00\x00\x1f\x00\x00\x00\x00serviceid\x00\x00\x00\x00\x00\x00\x00hostname\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x90x\x00\x00A'
  
  ### test ServiceLoginRespMessage
class ServiceLoginRespMessageTestCase(ConnectMessageTestCase):
    msgname = 'service_login_resp'
    items = (('status', 0x1234),)
    packet = '\x00\x00\x00\x10\xff\x00\x00\x1f\x00\x00\x00\x004\x12\x00\x00'

### test ServiceLogoutMessage
class ServiceLogoutMessageTestCase(ConnectMessageTestCase):
    msgname = 'service_logout'
    items = (('service_id', 'serviceid'),)
    packet = '\x00\x00\x00\x1c\x00\x00\x00 \x00\x00\x00\x00serviceid\x00\x00\x00\x00\x00\x00\x00'

### test ServiceLogoutRespMessage
class ServiceLogoutRespMessageTestCase(ConnectMessageTestCase):
    msgname = 'service_logout_resp'
    items = (('status', 0x1234),)
    packet = '\x00\x00\x00\x10\xff\x00\x00 \x00\x00\x00\x004\x12\x00\x00'

### test SelectServiceMessage
class SelectServiceMessageTestCase(ConnectMessageTestCase):
    msgname = 'select_service'
    items = (('service_id', 'serviceid'),)
    packet = '\x00\x00\x00\x1c\x00\x00\x00!\x00\x00\x00\x00serviceid\x00\x00\x00\x00\x00\x00\x00'

### test SelectServiceRespMessage
class SelectServiceRespMessageTestCase(GetApplicationRespMessageTestCase):
    msgname = 'select_service_resp'
    items = (('status', 0x1234),
             ('service_num', 0x0002),
            )
    resplist = ({'service_id': 's01',
                 'servicegroup_id': 'sg02',
                 'host': 'host01',
                 'port': 0x2345,
                 'level': 0x0001,
                 'weight': 0x0002,
                },
                {'service_id': 's02',
                 'servicegroup_id': 'sg02',
                 'host': 'host09',
                 'port': 0x2345,
                 'level': 0x0003,
                 'weight': 0x0004,
                },
               )
    packet = '\x00\x00\x00\xac\xff\x00\x00!\x00\x00\x00\x004\x12\x00\x00\x02\x00\x00\x00s01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00sg02\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00host01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00E#\x00\x00\x01\x00\x00\x00\x02\x00\x00\x00s02\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00sg02\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00host09\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00E#\x00\x00\x03\x00\x00\x00\x04\x00\x00\x00'

### test ServiceWaaringMessage
class ServiceWarningMessageTestCase(ConnectMessageTestCase):
    msgname = 'service_warning'
    items = (('service_id', 'serviceid'),
             ('warning_id', 'be warned sid'),
             ('level', 0x0001),
            )
    packet = '\x00\x00\x000\x00\x00\x00"\x00\x00\x00\x00serviceid\x00\x00\x00\x00\x00\x00\x00be warned sid\x00\x00\x00\x01\x00\x00\x00'

### test ServiceWarningRespMessage
class ServiceWarningRespMessageTestCase(ConnectMessageTestCase):
    msgname = 'service_warning_resp'
    items = (('status', 0x1234),)
    packet = '\x00\x00\x00\x10\xff\x00\x00"\x00\x00\x00\x004\x12\x00\x00'

### test ServiceNotifyMessage
class ServiceNotifyMessageTestCase(ConnectMessageTestCase):
    msgname = 'service_notify'
    items = (('service_id', 'serviceid'),
             ('level', 0x0001),
             ('weight', 0x0002),
             ('status', 'A'),
            )
    packet = '\x00\x00\x00%\x00\x00\x00#\x00\x00\x00\x00serviceid\x00\x00\x00\x00\x00\x00\x00\x01\x00\x00\x00\x02\x00\x00\x00A'
  
### test ServiceNotifyRespMessage
class ServiceNotifyRespMessageTestCase(ConnectMessageTestCase):
    msgname = 'service_notify_resp'
    items = (('status', 0x1234),)
    packet = '\x00\x00\x00\x10\xff\x00\x00#\x00\x00\x00\x004\x12\x00\x00'

### test AddGroup2AppMessage
class AddGroup2AppMessageTestCase(ConnectMessageTestCase):
    msgname = 'add_group2app'
    items = (('application_id', 'appid'),
             ('servicegroup_id', 'sgid'),
            )
    packet = '\x00\x00\x00,\x00\x00\x00$\x00\x00\x00\x00appid\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00sgid\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00'

### test AddGroup2AppRespMessage
class AddGroup2AppRespMessageTestCase(ConnectMessageTestCase):
    msgname = 'add_group2app_resp'
    items = (('status', 0x1234),)
    packet = '\x00\x00\x00\x10\xff\x00\x00$\x00\x00\x00\x004\x12\x00\x00'

### test DelGroup2AppMessage
class DelGroup2AppMessageTestCase(ConnectMessageTestCase):
    msgname = 'del_group2app'
    items = (('application_id', 'appid'),
             ('servicegroup_id', 'sgid'),
            )
    packet = '\x00\x00\x00,\x00\x00\x00%\x00\x00\x00\x00appid\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00sgid\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00'

### test DelGroup2AppRespMessage
class DelGroup2AppRespMessageTestCase(ConnectMessageTestCase):
    msgname = 'del_group2app_resp'
    items = (('status', 0x1234),)
    packet = '\x00\x00\x00\x10\xff\x00\x00%\x00\x00\x00\x004\x12\x00\x00'

### test GetGroup2AppMessage
class GetGroup2AppMessageTestCase(ConnectMessageTestCase):
    msgname = 'get_group2app'
    items = (('application_id', 'appid'),)
    packet = '\x00\x00\x00\x1c\x00\x00\x00&\x00\x00\x00\x00appid\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00'

### test GetGroup2AppRespMessage
class GetGroup2AppRespMessageTestCase(GetApplicationRespMessageTestCase):
    msgname = 'get_group2app_resp'
    items = (('status', 0x1234),
             ('servicegroup_num', 0x0002),
             )
    resplist = ({'servicegroup_id': 'sg01',
                 'desc': 'desc of sg01'
                },
                {'servicegroup_id': 'sg02',
                 'desc': 'desc of sg02'
                },
               )
    packet = '\x00\x00\x00\xb4\xff\x00\x00&\x00\x00\x00\x004\x12\x00\x00\x02\x00\x00\x00sg01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00desc of sg01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00sg02\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00desc of sg02\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00'

### test GetGroupOfAppMessage
class GetGroupOfAppMessageTestCase(ConnectMessageTestCase):
    msgname = 'get_groupofapp'
    items = (('servicegroup_id', 'sgid'),)
    packet = "\x00\x00\x00\x1c\x00\x00\x00'\x00\x00\x00\x00sgid\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"

### test GetGroupOfAppRespMessage
class GetGroupOfAppRespMessageTestCase(GetApplicationRespMessageTestCase):
    msgname = 'get_groupofapp_resp'
    items = (('status', 0x1234),
             ('application_num', 0x0002),
            )
    resplist = ({'application_id': 'app01',
                 'desc': 'desc of app01'
                },
                {'application_id': 'app02',
                 'desc': 'desc of app02'
                },
               )
    packet = "\x00\x00\x00\xb4\xff\x00\x00'\x00\x00\x00\x004\x12\x00\x00\x02\x00\x00\x00app01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00desc of app01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00app02\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00desc of app02\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"

### test AddService2GroupMessage
class AddService2GroupMessageTestCase(ConnectMessageTestCase):
    msgname = 'add_service2group'
    items = (('servicegroup_id', 'sgid'),
             ('service_id', 'serviceid'),
             ('level', 0x0001),
             ('weight', 0x0002),
            )
    packet = '\x00\x00\x004\x00\x00\x00(\x00\x00\x00\x00sgid\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00serviceid\x00\x00\x00\x00\x00\x00\x00\x01\x00\x00\x00\x02\x00\x00\x00'

### test AddService2GroupRespMessageTestCase
class AddService2GroupRespMessageTestCase(ConnectMessageTestCase):
    msgname = 'add_service2group_resp'
    items = (('status', 0x1234),)
    packet = '\x00\x00\x00\x10\xff\x00\x00(\x00\x00\x00\x004\x12\x00\x00'

### test DelService2GroupMessage
class DelService2GroupMessageTestCase(ConnectMessageTestCase):
    msgname = 'del_service2group'
    items = (('servicegroup_id', 'sgid'),
             ('service_id', 'serviceid'),
            )
    packet = '\x00\x00\x00,\x00\x00\x00)\x00\x00\x00\x00sgid\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00serviceid\x00\x00\x00\x00\x00\x00\x00'

### test DelService2GroupRespMessage
class DelService2GroupRespMessageTestCase(ConnectMessageTestCase):
    msgname = 'del_service2group_resp'
    items = (('status', 0x1234),)
    packet = '\x00\x00\x00\x10\xff\x00\x00)\x00\x00\x00\x004\x12\x00\x00'

### test ModifyService2GroupMessage
class ModifyService2GroupMessageTestCase(ConnectMessageTestCase):
    msgname = 'modify_service2group'
    items = (('servicegroup_id', 'sgid'),
             ('service_id', 'serviceid'),
             ('level', 0x0001),
             ('weight', 0x0002),
            )
    packet = '\x00\x00\x004\x00\x00\x00*\x00\x00\x00\x00sgid\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00serviceid\x00\x00\x00\x00\x00\x00\x00\x01\x00\x00\x00\x02\x00\x00\x00'

### test ModifyService2GroupRespMessage
class ModifyService2GroupRespMessageTestCase(ConnectMessageTestCase):
    msgname = 'modify_service2group_resp'
    items = (('status', 0x1234),)
    packet = '\x00\x00\x00\x10\xff\x00\x00*\x00\x00\x00\x004\x12\x00\x00'

### test GetService2GroupMessage
class GetService2GroupMessageTestCase(ConnectMessageTestCase):
    msgname = 'get_service2group'
    items = (('servicegroup_id', 'sgid'),)
    packet = '\x00\x00\x00\x1c\x00\x00\x00+\x00\x00\x00\x00sgid\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00'

### test GetService2GroupRespMessage
class GetService2GroupRespMessageTestCase(GetApplicationRespMessageTestCase):
    msgname = 'get_service2group_resp'
    items = (('status', 0x1234),
             ('service_num', 0x0002),
             ('list_num', 0x0002),
            )
    resplist = ({'service_id': 's01',
                 'host': 'host01',
                 'port': 0x2345,
                 'listen_host': 'host01',
                 'listen_port': 0x3456,
                 'level': 0x0001,
                 'weight': 0x0002,
                 'domain': 0x0003,
                 'tactics': 0x0004,
                 'status': 'A',
                },
                {'service_id': 's02',
                 'host': 'host02',
                 'port': 0x2345,
                 'listen_host': 'host02',
                 'listen_port': 0x3456,
                 'level': 0x0001,
                 'weight': 0x0002,
                 'domain': 0x0003,
                 'tactics': 0x0004,
                 'status': 'A',
                },
               )
    packet = '\x00\x00\x00\xea\xff\x00\x00+\x00\x00\x00\x004\x12\x00\x00\x02\x00\x00\x00\x02\x00\x00\x00s01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00host01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00E#\x00\x00host01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00V4\x00\x00\x01\x00\x00\x00\x02\x00\x00\x00\x03\x00\x00\x00\x04\x00\x00\x00As02\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00host02\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00E#\x00\x00host02\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00V4\x00\x00\x01\x00\x00\x00\x02\x00\x00\x00\x03\x00\x00\x00\x04\x00\x00\x00A'

### test GetServiceOfGroupMessage
class GetServiceOfGroupMessageTestCase(ConnectMessageTestCase):
    msgname = 'get_serviceofgroup'
    items = (('service_id', 'serviceid'),)
    packet = '\x00\x00\x00\x1c\x00\x00\x00,\x00\x00\x00\x00serviceid\x00\x00\x00\x00\x00\x00\x00'

### test GetServiceOfGroupRespMessage
class GetServiceOfGroupRespMessageTestCase(GetApplicationRespMessageTestCase):
    msgname = 'get_serviceofgroup_resp'
    items = (('status', 0x1234),
             ('servicegroup_num', 0x0002),
            )
    resplist = ({'servicegroup_id': 'sg01',
                 'desc': 'desc of sg01',
                },
                {'servicegroup_id': 'sg02',
                 'desc': 'desc of sg02',
                },
               )
    packet = '\x00\x00\x00\xb4\xff\x00\x00,\x00\x00\x00\x004\x12\x00\x00\x02\x00\x00\x00sg01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00desc of sg01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00sg02\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00desc of sg02\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00'
