# -*- coding: utf-8; test-case-name: compass.test.test_gnspmsg -*-
#
#
# $Id: gnspmsg.py,v 1.5 2004/11/15 07:03:09 xyb Exp $

"""Open Global Name Service Protocol

Stability: None

@author: U{HD<mailto:hdcola@gmail.com>}

@see: Open Global Name Service Protocol规范
"""
from compass.uss.message import bytemsg
import struct

class GNSPMessage(bytemsg.ByteMessage):
    """GNSSP 消息基类"""
    def __init__(self):
        # 消息头
        self.head = GNSPMessageHead(self)
        # 消息体
        self.body = bytemsg.ByteMessageBody(self)
        # 消息工具
        self.msgutilcls = GNSMessageUtil
        # 当前协议的名称
        self.protocolname = 'gnsp'
        # 当前的消息名称
        self.msgname = ''

    def pack_gnsp_connect(self, fields):
        """connect报文打包"""
        return struct.pack('6s16sI8s',
                fields['system_id'],
                fields['auth_source'],
                fields['version'],
                fields['time_stamp']
                )

    def unpack_gnsp_connect(self, packet):
        """connect报文解包"""
        self.body.fields['system_id'], \
            self.body.fields['auth_source'], \
            self.body.fields['version'], \
            self.body.fields['time_stamp'] = \
            struct.unpack('6s16sI8s',packet)

    def pack_gnsp_connect_resp(self, fields):
        """connect_resp报文打包"""
        return struct.pack('II',
                fields['status'],
                fields['version']
                )

    def unpack_gnsp_connect_resp(self, packet):
        """connect_resp报文解包"""
        self.body.fields['status'], \
            self.body.fields['version'] = \
            struct.unpack('II',packet)

    def pack_gnsp_mail_counter(self, fields):
        """mail_counter报文打包"""
        return struct.pack('16s', fields['uid'])

    def unpack_gnsp_mail_counter(self, packet):
        """mail_counter报文解包"""
        (self.body.fields['uid'],) = struct.unpack('16s',packet)

    def pack_gnsp_mail_counter_resp(self, fields):
        """mail_counter_resp报文打包"""
        return struct.pack('16sI',
                fields['uid'],
                fields['number']
                )

    def unpack_gnsp_mail_counter_resp(self, packet):
        """mail_counter_resp报文解包"""
        self.body.fields['uid'], \
            self.body.fields['number'] = \
            struct.unpack('16sI', packet)

    def pack_gnsp_terminate(self, fields):
        """terminate报文体为空"""
        return None

    def unpack_gnsp_terminate(self, packet):
        """terminate报文体为空"""
        return None

    def pack_gnsp_terminate_resp(self, fields):
        """terminate_resp报文体为空"""
        return None

    def unpack_gnsp_terminate_resp(self, packet):
        """terminate_resp报文体为空"""
        return None

class GNSPMessageHead(bytemsg.ByteMessageHead):
    """GNSP消息的消息头抽像类"""
    def __init__(self, parent):
        # 长度
        self.length = 8
        # 命令ID
        self.id = 0
        # 他爹
        self.parent = parent

    def __len__(self):
        """包头定长8"""
        return 8

    def setId(self,id):
        """设置命令ID"""
        self.id = id

    def setLength(self, length):
        """包的整长"""
        self.length = length

    def loadHead(self, header):
        """转换一个PDU (protocol data unit)到一个消息头
        消息头的格式为length(4字节)、commandid（4字节），共8字节长度
        """
        self.length,self.id = struct.unpack('>II', header)

    def packed(self):
        """转换一个消息头为二进制流
        消息头的格式为length(4字节)、commandid（4字节，共12字节长度
        """
        return struct.pack('>II', self.length, self.id)

    def __str__(self):
        """字符串化"""
        plist = []
        plist.append("消息ID:%s" % self.id)
        plist.append("消息长度：%s" % self.length)
        return reduce(lambda x,y: x + "\n" + y, plist)

class GNSMessageUtil(bytemsg.ByteMessageUtil):
    """GNSSP消息处理工具类"""
    def __init__(self):
        pass

# GNSSP Message定义
GNSMessageUtil.commandinfo = {
    0x00000000L : 'generic_noop',        # 什么都不做的报文
    0xff000000L : 'generic_noop_resp',   # 什么都不做的报文回复
    0x00000001L : 'connect',             # 连接请求报文
    0xff000001L : 'connect_resp',        # 连接请求报文回复
    0x00000002L : 'terminate',           # 断开连接请求报文
    0xff000002L : 'terminate_resp',      # 断开连接请求报文回复
    0x00000005L : 'mail_counter',        # 邮件统计报文
    0xff000005L : 'mail_counter_resp'    # 邮件统计报文回复
    }

# 通过名称查出消息ID的结构定义
GNSMessageUtil.nametoid = {}
for k in GNSMessageUtil.commandinfo.keys():
    GNSMessageUtil.nametoid[GNSMessageUtil.commandinfo[k]] = k
